' ------------------------------------------------------------------------------------------------
' Milkyway Type
' ------------------------------------------------------------------------------------------------
Type TMilkyway



	Field masterpivot:TPivot

	' Variables
	Field Seed:Int = 42											' Base seed to recreate a scene
	Field Stars:Int = 10000										' Number of fixed stars
	Field MaxAngle:Float = 45									' Maximum angle of Star creation
	Field Scale:Float = 512										' Scale of Scene
	Field Height:Float = Scale * 0.75							' Height of Panorama
	Field Segments:Int = 30										' Panorama Mesh Segments
	Field StarScale:Float = 2.0									' Star size multiplicator
	Field FixedStars:Float = 1.0 / 25							' Amount of larger stars (0...1)
	
	' Colors
	Field PanoRGB1:Int[] = [0, 0, 0, 0]							' Panorama upper vertex colors
	Field PanoRGB2:Int[] = [64, 64, 64, 255]					' Panorama center vertex colors
	Field PanoRGB3:Int[] = [0, 0, 0, 0]							' Panorama lower vertex colors

	' Star
	Field StarsRGB1:Int[] = [0, 0, 0, 0]						' Stars upper vertex colors
	Field StarsRGB2:Int[] = [192, 192, 192, 128]				' Stars center vertex colors
	Field StarsRGB3:Int[] = [0, 0, 0, 0]						' Stars upper vertex colors
	
	' Detail
	Field DetailRGB1:Int[] = [0, 0, 0, 0]						' Detail upper vertex colors
	Field DetailRGB2:Int[] = [64, 64, 128, 255]				' Detail center vertex colors
	Field DetailRGB3:Int[] = [0, 0, 0, 0]						' Detail upper vertex colors
	
	' Entities
	Field Pivot:TPivot
	Field StarSphere:TMesh
	Field Panorama:TMesh
	Field Detail:TMesh
	Field StarDetail:TMesh
	
	' Textures
	Field StarSphereTexture:TTexture
	Field PanoramaTexture:TTexture
	Field DetailTexture:TTexture
	Field Brightness:TTexture
	Field StarsTexture:TTexture
		
	' Entity FX
	Field FX_StarSphere:Int = 1 + 2 + 8 + 32
	Field FX_Panorama:Int = 1 + 2 + 8 + 32
	Field FX_StarDetail:Int = 1 + 2 + 8 + 32
	Field FX_Detail:Int = 1 + 2 + 8 + 32
	
	' Entity Blendings
	Field BLEND_StarSphere:Int = 3
	Field BLEND_Panorama:Int = 1
	Field BLEND_StarDetail:Int = 3
	Field BLEND_Detail:Int = 3
	
	' Texture Blendings
	Field TEXBLEND_StarSphere:Int = 3
	Field TEXBLEND_Panorama:Int = 5
	Field TEXBLEND_StarDetail:Int = 5
	Field TEXBLEND_Detail:Int = 5
			
	Field r:Int, g:Int, b:Int
	
	Field star:TQuad
	Field starlist:TList



	' --------------------------------------------------------------------------------------------
	' Initialize Milkyway Type
	' --------------------------------------------------------------------------------------------
	Method Init()
						
		' Main Pivot
		PIVOT = CreatePivot(masterpivot)
					
		' Create Meshes
		CreateStarSphere(PIVOT)
		Panorama = CreatePanorama(PIVOT, PanoRGB1, PanoRGB2, PanoRGB3)
		StarDetail = CreatePanorama(PIVOT, StarsRGB1, StarsRGB2, StarsRGB3)
		Detail = CreatePanorama(PIVOT, DetailRGB1, DetailRGB2, DetailRGB3)
											
		' Entity FX
		EntityFX StarSphere, FX_StarSphere
		EntityFX Panorama, FX_Panorama
		EntityFX StarDetail, FX_StarDetail
		EntityFX Detail, FX_Detail
		
		EntityOrder Panorama, 1000
		EntityOrder Detail, 750
		EntityOrder StarDetail, 500
		EntityOrder Starsphere, 250
			
		' Entity Blending
		EntityBlend StarSphere, BLEND_StarSphere
		EntityBlend Panorama, BLEND_Panorama
		EntityBlend StarDetail, BLEND_StarDetail
		EntityBlend Detail, BLEND_Detail

		' Entity Texturing
		EntityTexture StarSphere, StarSphereTexture
		EntityTexture Panorama, PanoramaTexture
		EntityTexture StarDetail, StarsTexture
		EntityTexture Detail, DetailTexture
		
		ScaleTexture DetailTexture, 1.0 / 4, 1.0 / 1
		PositionTexture DetailTexture, Rnd(0, 1), Rnd(0, 1)

		ScaleTexture StarsTexture, 1.0 / 8, 1.0 / 2
		PositionTexture StarsTexture, Rnd(0, 1), Rnd(0, 1)
				
		' Texture Blending
		TextureBlend StarSphereTexture, TEXBLEND_StarSphere
		TextureBlend PanoramaTexture, TEXBLEND_Panorama
		TextureBlend DetailTexture, TEXBLEND_Detail
		TextureBlend StarsTexture, TEXBLEND_StarDetail
									
		ScaleTexture DetailTexture, 1.0 / 8, 1.0 / 2
		ScaleTexture StarsTexture, 1.0 / 32, 1.0 / 8
		
	End Method

		

	' --------------------------------------------------------------------------------------------
	' Remove Milkyway Type
	' --------------------------------------------------------------------------------------------
	Method Remove()
	
		If Panorama Then FreeEntity Panorama
		If StarSphere Then FreeEntity StarSphere
		If Detail Then FreeEntity Detail
		If PIVOT Then FreeEntity PIVOT
						
	End Method



	' --------------------------------------------------------------------------------------------
	' Update Panorama Vertex Colors
	' --------------------------------------------------------------------------------------------
	Method UpdatePanoramaColor(rgb1:Int[], rgb2:Int[], rgb3:Int[])
	
		Local surf:TSurface = GetSurface(Panorama, 1)
				
		For Local v:Int = 0 To CountVertices(surf) - 1 Step 6

			VertexColor surf, v + 0, RGB1[0], RGB1[1], RGB1[2], RGB1[3] / 255.0
			VertexColor surf, v + 1, RGB1[0], RGB1[1], RGB1[2], RGB1[3] / 255.0
			VertexColor surf, v + 2, RGB2[0], RGB2[1], RGB2[2], RGB2[3] / 255.0
			VertexColor surf, v + 3, RGB2[0], RGB2[1], RGB2[2], RGB2[3] / 255.0
			VertexColor surf, v + 4, RGB3[0], RGB3[1], RGB3[2], RGB3[3] / 255.0
			VertexColor surf, v + 5, RGB3[0], RGB3[1], RGB3[2], RGB3[3] / 255.0
		
		Next
			
	End Method
	
	
	
	' --------------------------------------------------------------------------------------------
	' Milkyway Panorama
	' --------------------------------------------------------------------------------------------
	Method CreatePanorama:TMesh(parent:TEntity = Null, rgb1:Int[], rgb2:Int[], rgb3:Int[])
		
		Local a1:Float, a2:Float, inc:Float, x1:Float, z1:Float, x2:Float, z2:Float
		Local uu1:Float, uu2:Float, uv1:Float, uv2:Float, uv3:Float
		Local v0:Int, v1:Int, v2:Int, v3:Int, v4:Int, v5:Int
		
		Local mesh:TMesh = CreateMesh(parent)
		Local surf:TSurface = CreateSurface(mesh)
			
		Inc = 360.0 / SEGMENTS
			
		While a1 < 360
				
			a2 = (a1 + Inc) Mod 360
					
			x1 = Scale * Cos(a1)
			z1 = Scale * Sin(a1)
			x2 = Scale * Cos(a2)
			z2 = Scale * Sin(a2)

			uu1 = 1.0 - (a1 / 360.0)
			uu2 = 1.0 - (a1 + Inc) / 360.0

			uv1 = 1.00
			uv2 = 0.50
			uv3 = 0.00
			
			v0 = AddVertex(surf, x1, -Height, z1, uu1, uv1)
			v1 = AddVertex(surf, x2, -Height, z2, uu2, uv1)
			v2 = AddVertex(surf, x1, 0, z1, uu1, uv2)
			v3 = AddVertex(surf, x2, 0, z2, uu2, uv2)
			v4 = AddVertex(surf, x1, Height, z1, uu1, uv3)
			v5 = AddVertex(surf, x2, Height, z2, uu2, uv3)
			
			VertexColor surf, v0, RGB1[0], RGB1[1], RGB1[2], RGB1[3] / 255.0
			VertexColor surf, v1, RGB1[0], RGB1[1], RGB1[2], RGB1[3] / 255.0
			VertexColor surf, v2, RGB2[0], RGB2[1], RGB2[2], RGB2[3] / 255.0
			VertexColor surf, v3, RGB2[0], RGB2[1], RGB2[2], RGB2[3] / 255.0
			VertexColor surf, v4, RGB3[0], RGB3[1], RGB3[2], RGB3[3] / 255.0
			VertexColor surf, v5, RGB3[0], RGB3[1], RGB3[2], RGB3[3] / 255.0
			
			AddTriangle surf, v0, v1, v3
			AddTriangle surf, v3, v2, v0
			AddTriangle surf, v2, v5, v4
			AddTriangle surf, v5, v2, v3
			
			a1:+Inc
					
		Wend
		
		Return mesh
				
	End Method

	
	
	' --------------------------------------------------------------------------------------------
	' Fixed Stars
	' --------------------------------------------------------------------------------------------
	Method CreateStarSphere:TMesh(parent:TEntity = Null)
		
		Local i:Int
		Local surf:TSurface
		Local col:Float
		Local angle:Float
		Local f:Float
		Local cat:Int
		
		StarSphere = CreateMesh(parent)
		surf = CreateSurface(StarSphere)
		starlist = CreateList()
						
		For i = 1 To Stars
		
			Star = New TQuad
			Star.mesh = StarSphere
			Star.surf = surf
			Star.ScaleX = Rnd(Rnd(0.2, 0.4), Rnd(0.4, 0.6)) * STARSCALE
			angle = Rnd(-MaxAngle, MaxAngle)
			
			If Rnd(1) > (1 - FixedStars) Then
			
				Star.ScaleX = Rnd(Rnd(0.5, 1.0), Rnd(1.0, Rnd(1.25, Rnd(1.5, 2)))) * StarScale
				angle = MaxAngle
			
			EndIf
			
			' Y scale
			Star.ScaleY = Star.ScaleX
						
			' Spectral Class spread
			col = Rnd(1)
			If col > 0.92 And col <= 1.00 Then cat = 0
			If col > 0.85 And col <= 0.92 Then cat = 1
			If col > 0.78 And col <= 0.85 Then cat = 2
			If col > 0.71 And col <= 0.78 Then cat = 3
			If col > 0.64 And col <= 0.71 Then cat = 4
			If col > 0.57 And col <= 0.64 Then cat = 5
			If col > 0.50 And col <= 0.57 Then cat = 6
			If col > 0.00 And col <= 0.50 Then cat = 7

			' Spectral Class color and scale
			Select cat
				
				Case 0 r = 096 ; g = 104 ; b = 112 ; f = 0.1		' W
				Case 1 r = 080 ; g = 088 ; b = 255 ; f = 1.5		' O
				Case 2 r = 080 ; g = 128 ; b = 255 ; f = 1.25		' B
				Case 3 r = 100 ; g = 100 ; b = 100 ; f = 1.0		' A
				Case 4 r = 128 ; g = 100 ; b = 080 ; f = 0.9		' F
				Case 5 r = 128 ; g = 080 ; b = 040 ; f = 0.8		' G
				Case 6 r = 255 ; g = 064 ; b = 000 ; f = 0.7		' K
				Case 7 r = 064 ; g = 016 ; b = 000 ; f = 0.6		' M
				
			End Select
			
			Star.scalex:*f
			Star.scaley:*f
						
			Star.RGB = [r * 2, g * 2, b * 2]
										
			Star.x = Rnd(-100, 100)
			Star.y = Rnd(-angle, angle)
			Star.z = Rnd(-100, 100)
					
			TFormNormal Star.x, Star.y, Star.z, Null, Null
			
			Local d:Float = Rnd(1.0, 1.5)
			
			Star.x = TFormedX() * Scale * d
			Star.y = TFormedY() * Scale * d
			Star.z = TFormedZ() * Scale * d
				
			Star.Add()
			Star.AlignToTarget(Star.x, Star.y, Star.z, 0, 1)
			
			ListAddLast(starlist, Star)
					
		Next
		
	End Method
	
	
	
	' --------------------------------------------------------------------------------------------
	' Hold position
	' --------------------------------------------------------------------------------------------
	Method KeepPosition(parent:TEntity)

		PositionEntity PIVOT, EntityX(parent, 1), EntityY(parent, 1), EntityZ(parent, 1)

	End Method

	

End Type