﻿' single Planet in TSystem
Type TPlanet

	Field Planets:Int=10

	Field id:String							' Unique Planet ID
	Field DistanceAU:Double					' Planet Distance to Start in AU
	Field DistanceKM:Double					' Planet Distance to Start in Mio. Kilometers
	Field Mass:Double						' Planet Mass in kg
	Field Density:Double					' Planet Density in g/cm³
	Field Volume:Double						' Planet Volume in cm³
	Field Radius:Double						' Planet Radius in km
	Field Category:String					' Planet category
	Field SizeCategory:String				' Planet size category
	Field TemperatureCategory:String		' Planet temperature category
	Field EscapeVelocity:Double				' Planet Escape Velocity in km/s
	Field OrbitalSpeed:Double				' Orbital Speed in km/s
	Field SurfaceGravity:Double				' Surface Gravity in Earth g
	Field AtmoPressure:Double				' Atmospheric Pressure in bar
	Field SidericDays:Double				' Sideric Period Days
	Field SidericYears:Double				' Sideric Period Years
	Field Rotationspeed:Double				' Rotation Speed per day
	
	Field AlbedoAlpha:Double				' Planet Albedo #1
	Field AlbedoBeta:Double					' Planet Albedo #2
	Field AlbedoTotal:Double				' Total Planet Albedo
	Field SolarConstant:Double				' Planet Solar Constant in Watt/s per square meter
	Field SolarCorrectedConstant:Double		' Corrected Planet Solar Constant in Watt/s per square meter atmosphere
	Field SurfaceEnergy:Double				' Energy on 1 square meter surface in Watt/s
	Field TemperatureKelvin:Double			' Average temperature in Kelvin
	Field TemperatureCelsius:Double			' Average temperature in Celsius
	Field GreenHouseEffect:Double			' Greenhouse effect
	Field TidallyLocked:Int					' Flag if planet is tidally locked to its star
	Field TidallyLockedTime:Double			' Time until a planet gets tidally locked
	Field TemperatureMin:Double				' Minimum Temperature in Kelvin
	Field TemperatureMax:Double				' Maximum Temperature in Kelvin
	Field TemperatureMinCorrected:Double	' Minimum Corrected Temperature in Kelvin
	Field TemperatureMaxCorrected:Double	' Maximum Corrected Temperature in Kelvin
	Field TemperatureDifference:Double		' Temperature Difference highest/lowest
	Field Temperature:Double				' Average Planet Temperature in Celsius

	Field ElementMass:Double				' help variable to determine heaviest element / Litosphere
	Field MaximumAtmosphereMass:Double		' help variable to determine Atmosphere composition
	Field MaximumHydrosphereMass:Double		' help variable to determine Hydrosphere composition
	Field MaximumLithosphereMass:Double		' help variable to determine heaviest Litosphere element

	Field Atmosphere:String					' Atmosphere Compound
	Field Hydrosphere:String				' Hydrosphere Compound
	Field Lithosphere:String				' Lithosphere Compound
	Field Weather:String					' Global Weather situation

	Field Resources:Double					' probability to find resources on this planet in percent

	Field InsideHabitableZone:Int			' Bonus Evalution Score field: planet is inside the star's habitable zone
	Field SuitableClimateZone:Int			' Bonus Evalution Score field: planet is in the star's proper temperature zone
	Field SuitablePlanetCategory:Int		' Bonus Evalution Score field: planet category is suitable for life

	Field PlanetNotTidalLocked:Int			' Standard Evaluation Score field: planet is still rotation = balanced weather
	Field StarSuitableForLife:Int			' Standard Evaluation Score field: star has no lethal radiation = M,K,G,F stars
	Field SuitableDensity:Int				' Standard Evaluation Score field: density is high enough for wide spread elements
	Field SuitableGForce:Int				' Standard Evaluation Score field: gravity is not too low and not too high
	Field SuitableAtmospherePressure:Int	' Standard Evaluation Score field: atmosphere pressure is not too low and not too high
	Field SuitableRotationSpeed:Int			' Standard Evaluation Score field: rotation speed is not too low and not too high
	Field SuitableTemperature:Int			' Standard Evaluation Score field: average temperature is not too low and not too high
	Field PlateTectonics:Int				' Standard Evaluation Score field: planet has plate tectonics = recycling of elements

	Field EvaluationScore:Double			' weighted sum of all Evaluation score fields = habitability of this planet
	Field ESI:Double						' Earth Similarity Index
	Field ESIRadius:Double					' Size part of ESI
	Field ESIDensity:Double					' Density part of ESI
	Field ESITemperature:Double				' Temperature part of ESI
	Field ESICosmicSpeed:Double				' Gravitation part of ESI
	Field ESIInterior:Double				' Interior part of ESI
	Field ESISurface:Double					' Surface part of ESI
	
	
	Field TempID:Int
	Field SizeID:Int

	Method CalcDistanceAU:Double(System:TSystem, i:Int)

		Local multi:Double = Sqr(System.MassEstimated * 100) * 1.0 / System.MassEstimated
		
		' special case White Dwarf
		If System.SpectralType = "W" Then Return (0.4 + 0.3 * 2 ^ i) * (System.MassCalculated / multi)
		
		Return ((0.4 + (0.3 * 2 ^ (i - 0.75))) * System.MassCalculated)
				
		'Return (System.MinimumPlanetDistance + (0.4 + 0.3 * 2 ^ (i - 1))) * System.MassCalculated
		'Return 0.4 + (0.3 * (2 ^ (i-1)))' * System.MassCalculated
		
		'Return (4 + (3 * (2 ^ (i - 2)))) / 10.0 * System.MassCalculated
		
		'Return (((Sqr(System.HZMinMinimumPlanetDistanceAU)*System.MassTypical)+((0.125*2^(i-1))*System.HZMinMinimumPlanetDistanceAU))+wd)*multi

	End Method

	Method CalcMass:Double(System:TSystem, i:Int)
	
		'Return (Rnd(10) + 1) * 10.0 ^ Rand(System.Masses[i - 1, 1], System.Masses[i - 1, 2])
		Return (Rnd(10) + 1) * 10.0 ^ Rand(21, 29)

	End Method

	Method CalcRotationSpeed:Double()

		If TidallyLocked = 0 Then Return (Log10(SidericDays) * 1.0 / SidericYears) Else Return 0
		
	End Method

	Method IsTidallyLocked:Int(System:TSystem)

		If TidallyLockedTime <= System.Age Then Return 1 Else Return 0

	End Method

	Method CalcTidallyLockedTime:Double(System:TSystem)

		Local a1:Double = (6.0 * ((DistanceKM * 1000000000.0) ^ 6) * Radius * 1000.0 * (3.0 * (10.0 ^ 10)))
		Local a2:Double = Mass * (System.Mass ^ 2)
		Local a3:Double = a1 * 1.0 / a2 * 10.0 ^ 10

		Return a3

	End Method

	Method CalcGreenhouseEffect:Double(i:Int)

		Local g1:Double = Sqr(SurfaceEnergy * (1 - AlbedoTotal)) * (planets - i) / 2.0 * (1 + (TidallyLocked * (1 - AlbedoTotal)))
		Local g2:Double = 0.0

		If SizeCategory = "Brown Dwarf" Then g2 = Sqr(Mass * 1.0 / 10 ^ 27 * Sqr(Radius))' ; Print i + ": Brown Dwarf"

		Return (g1 + g2)

	End Method

	' calculate planet size category according to radius
	Method CalcSizeCategory()

		RestoreData Size

		Local d:String[6, 3]

		For Local r:Int = 0 To 5

			For Local c:Int = 0 To 2

				ReadData d[r, c]

			Next

			If Radius >= Double(d[r, 0]) And Radius < Double(d[r, 1]) Then SizeCategory = d[r, 2] ; SizeID = r

		Next

	End Method

	' calculate planet temperature category according to average temperature
	Method CalcTemperatureCategory()

		RestoreData Temperature

		Local d:String[7, 3]

		For Local r:Int = 0 To 6

			For Local c:Int = 0 To 2

				ReadData d[r, c]

			Next

			If Temperature >= Double(d[r, 0]) And Temperature < Double(d[r, 1]) Then TemperatureCategory = d[r, 2] ; TempID = 6 - r

		Next

	End Method

	' calculate planet category according to size and average temperature
	Method CalcCategory()

		RestoreData Category

		Category = "None"

		Local d:String[42, 2]
		Local s:String = TemperatureCategory + " " + SizeCategory

		For Local r:Int = 0 To 41

			For Local c:Int = 0 To 1

				ReadData d[r, c]

			Next

			If s = d[r, 0] Then Category = d[r, 1]

		Next

	End Method

	Method CalcAtmosphere()

		RestoreData Atmosphere

		Atmosphere = "None"

		Local d:String[10, 4]

		For Local r:Int = 0 To 9

			For Local c:Int = 0 To 3

				ReadData d[r, c]

			Next
			
			'Print Density+" | "+d[r, 0] + " | " + d[r, 1]

			If Density >= Double(d[r, 0]) And Density < Double(d[r, 1]) Then

				Atmosphere = d[r, 2]
				MaximumAtmosphereMass = Double(d[r, 1])

			EndIf

		Next

	End Method

	Method CalcHydrosphere()

		RestoreData Hydrosphere

		Hydrosphere = "None"

		Local d:String[10, 4]

		For Local r:Int = 0 To 9

			For Local c:Int = 0 To 3

				ReadData d[r, c]

			Next

			If Temperature >= Double(d[r, 0]) And Temperature < Double(d[r, 1]) Then

				Hydrosphere = d[r, 2]
				MaximumHydrosphereMass = Double(d[r, 1])

			EndIf

		Next

	End Method

	Method CalcLithosphere()

		RestoreData Lithosphere

		Lithosphere = ""

		Local d:String[25, 4]

		For Local r:Int = 0 To 24

			For Local c:Int = 0 To 3

				ReadData d[r, c]

			Next

			If ElementMass >= Double(d[r, 0]) And ElementMass < Double(d[r, 1]) Then

				Lithosphere = d[r, 2]
				MaximumLithosphereMass = Double(d[r, 1])

			EndIf

		Next

	End Method

	Method CalcWeather()

		RestoreData Weather

		Weather = "-"

		Local d:String[6, 3]

		For Local r:Int = 0 To 5

			For Local c:Int = 0 To 2

				ReadData d[r, c]

			Next

			If TemperatureDifference >= Double(d[r, 0]) And TemperatureDifference < Double(d[r, 1]) Then

				Weather = d[r, 2]

			EndIf

		Next

	End Method

	Method CalcEvaluationScore(System:TSystem)

		Local e:Float = 0.0
		Local e1:Float = 0.0
		Local e2:Float = 0.0
		Local f:Int[] = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]
		Local t:Int = 0

		' check InsideHabitableZone
		If DistanceAU >= System.HZMinMinimumPlanetDistanceAU Then f[0]:+1
		If DistanceAU <= System.HZMaxMinimumPlanetDistanceAU Then f[0]:+1 Else f[0]:-1
		InsideHabitableZone = f[0] * System.PlanetEvaluationDistribution[0]

		' check SuitableClimateZone
		t = 0
		If TemperatureCategory = "Arid" Then t:+1
		If TemperatureCategory = "Temperare" Then t:+1
		If TemperatureCategory = "Cold" Then t:+1
		If t > 0 Then f[1]:+1 Else f[1]:-1

		' check SuitablePlanetCategory
		t = 0
		If SizeCategory = "Dwarf" Then t:+1
		If SizeCategory = "Rock" Then t:+1
		If SizeCategory = "Super Earth" Then t:+1
		If t > 0 Then f[2]:+1 Else f[2]:-1

		' check PlanetNotTidalLocked
		If TidallyLockedTime >= System.Age Then f[3]:+1
		
		' check StarSuitableForLife
		t = 0
		t:+System.StarOldEnough + System.SpectrumSuitable + System.MetallicitySuitable
		If t > 0 Then f[4]:+1

		' check SuitableDensity
		If Density >= System.MinimumDensity And Density <= System.MaximumDensity Then f[5]:+1

		' check SuitableGForce
		If SurfaceGravity >= System.MinimumGForce And SurfaceGravity <= System.MaximumGForce Then f[6]:+1

		' check SuitableAtmospherePressure
		If AtmoPressure >= System.MinimumAtmoPressure And AtmoPressure <= System.MaximumAtmoPressure Then f[7]:+1

		' check SuitableRotationSpeed
		If RotationSpeed >= System.MinimumRotationSpeed And RotationSpeed <= System.MaximumRotationSpeed Then f[8]:+1

		' check SuitableTemperature
		If Temperature >= System.MinimumHZTemperature And Temperature <= System.MaximumHZTemperature Then f[9]:+1

		' check PlateTectonics
		If Radius >= 6000 And Radius <= 12000 Then f[10]:+1

		For Local i:Int = 3 To 10

			e:+(f[i] * System.PlanetEvaluationDistribution[i])

		Next

		e1 = System.PlanetEvaluationDistribution[0] + System.PlanetEvaluationDistribution[1] + System.PlanetEvaluationDistribution[2]
		e2 = (f[0] * System.PlanetEvaluationDistribution[0]) + (f[1] * System.PlanetEvaluationDistribution[1]) + (f[2] * System.PlanetEvaluationDistribution[2])

		EvaluationScore = (e) - (e1) + (e2)

		If Evaluationscore < 0 Then Evaluationscore = 0
		If Evaluationscore > 100 Then Evaluationscore = 100
		
	End Method
	
	Method CalcESI:Double()
	
		Local RadiusEarth:Double = 6378.0
		Local DensityEarth:Double = 5.515
		Local TempAvgEarth:Double = 15.0' + 273.0
		Local CosmicSpeedEarth:Double = 7.909788019

		'Local MassEarth:Double = 5.974 * (10 ^ 24)
		'Local AlbedoEarth:Double = 0.367
		'Local GravityEarth:Double = 1.0
		'Local PressureEarth:Double = 1.014
		
		Local CosmicSpeed:Double = Sqr(0.0000000000667 * Mass / (Radius * 1000)) / 1000

		ESIRadius = (1 - Abs((Radius - RadiusEarth) / (Radius + RadiusEarth))) ^ 0.57
		ESIDensity = (1 - Abs((Density - DensityEarth) / (Density + DensityEarth))) ^ 1.07
		ESITemperature = (1 - Abs((Temperature - TempAvgEarth) / (Temperature + TempAvgEarth))) ^ 5.58
		ESICosmicSpeed = (1 - Abs((CosmicSpeed - CosmicSpeedEarth) / (CosmicSpeed + CosmicSpeedEarth))) ^ 0.7
		
		ESIInterior = Sqr(ESIRadius * ESIDensity)
		ESISurface = Sqr(ESITemperature * ESICosmicSpeed)
    
		ESI = Sqr(ESIInterior * ESISurface)
		If ESI < 0 Then ESI = 0
				
		Return ESI
	
	End Method

	Method CalcPlanetSystem(System:TSystem,id:Int=0)

		If id = 0 Then id=Rand(1000000000)

		System.Init()
		System.NewRandomSystem()
		System.CalcSystem()

		For Local i:Int = 1 To PLANETS
		
			System.Planet[i] = New TPlanet
		
			Local P:TPlanet = System.Planet[i]
		
			P.id = i
			P.DistanceAU = P.CalcDistanceAU(System, i)
						
			P.DistanceKM = P.DistanceAU * System.AstronomicalUnitKM / 1000000.0
			P.Mass = P.CalcMass(System, i)
			P.Density = Abs(Log10(P.Mass * 1.0 / 10.0 ^ 30.0)) / 1.25
			P.Volume = (P.Mass * 1000.0) / P.Density
			P.Radius = (Sqr(3.0 * P.Volume / (4.0 * Pi)) ^ 0.33333333) / 4.0
			P.EscapeVelocity = Sqr(System.GravitationalConstant * P.Mass / (P.Radius * 1000)) / 1000.0
			P.OrbitalSpeed = Sqr(System.GravitationalConstant * (System.Mass + P.Mass) / (P.DistanceKM * 1000000000000000.0))
			P.TidallyLockedTime = P.CalcTidallyLockedTime(System)
			P.TidallyLocked = P.IsTidallyLocked(System)
			P.SurfaceGravity = ((System.GravitationalConstant * (P.Mass / System.SunMass) * System.SunMass) / ((P.Radius * 1000) ^ 2)) / System.EarthGForce
			P.AtmoPressure = Sqr(P.SurfaceGravity) * Sqr(P.Density) ^ 0.5
			P.SidericDays = (P.DistanceAU * 365.25) + (Sqr(P.DistanceAU) * i ^ 2) * (1.0 / i)
			P.SidericYears = P.SidericDays / 365.25
			P.RotationSpeed = P.CalcRotationSpeed()
		
			P.AlbedoAlpha = Rnd(1) / 1.5
			P.AlbedoBeta = P.AlbedoAlpha / 2.0
			P.AlbedoTotal = P.AlbedoAlpha + P.AlbedoBeta

			P.SolarConstant = System.CorrectedHeatOutput / (4.0 * Pi * (P.DistanceKM * 1000000000.0) ^ 2)
			P.SolarCorrectedConstant = P.SolarConstant / 4.0
			'P.SurfaceEnergy = P.SolarCorrectedConstant * (1 - P.AlbedoAlpha + P.AlbedoBeta)
			P.SurfaceEnergy = (System.MegaWattPerSquareMeter * (10 ^ 6) * Double(System.StarSurface)) / (4 * Pi * Double(P.DistanceAU * System.AstronomicalUnitM) ^ 2)
				
			P.ElementMass = Sqr(P.Density * 25) * P.Density
			
			P.CalcAtmosphere()
			P.CalcHydrosphere()
			P.CalcLithosphere()

			P.GreenHouseEffect = P.CalcGreenHouseEffect(i)
			'P.TemperatureKelvin = ((P.SurfaceEnergy * 1.0 / System.Boltzmann) ^ 0.25) + P.GreenHouseEffect
			P.TemperatureKelvin = ((P.SurfaceEnergy / 4.0 / System.Boltzmann) ^ 0.25)

			P.TemperatureCelsius = P.TemperatureKelvin - System.CelsiusKelvin
			P.TemperatureMin = (P.TemperatureKelvin * Rnd(0.7, 0.9) * (1 + P.TidallyLocked))
			P.TemperatureMax = (P.TemperatureKelvin * Rnd(1.1, 1.3) * (1 + P.TidallyLocked))
			P.TemperatureMinCorrected = (P.TemperatureMin + P.GreenHouseEffect) - System.CelsiusKelvin
			P.TemperatureMaxCorrected = (P.TemperatureMax + P.GreenHouseEffect) - System.CelsiusKelvin
			P.Temperature = (P.TemperatureMinCorrected + P.TemperatureMaxCorrected) / 2.0
			P.TemperatureDifference = (P.TemperatureMaxCorrected - P.TemperatureMinCorrected) / 2 * (1 + P.TidallyLocked)
		
			P.CalcESI()
		
			P.CalcSizeCategory()
			P.CalcTemperatureCategory()
			P.CalcWeather()

			P.CalcCategory()

			P.Resources = Rnd(1) * Sqr(P.Density / i) / 2.5
			If P.Resources > 1 Then P.Resources = 1
		
			P.CalcEvaluationScore(System)

		Next

	End Method

End Type