﻿Type TCategory

	Global CategoryMap:TMap = CreateMap()

	Global rows:Int = 20
	Global cols:Int = 2

	Field Category:String
	Field Description:String

	' Read Category Field values
	Function Init()

		' Array to hold Data field values
		Local ca:String[cols]

		' access Data field
		RestoreData Category

		' cycle all rows
		For Local i:Int = 0 To rows - 1

			' cycle all columns
			For Local j:Int = 0 To cols - 1
		
				' store to array
				ReadData ca[j]

			Next

			' create a new TStarData object
			TCategory.Create(ca)
		
		Next

	End Function

	' create a single Category Object
	Function Create(ca:String[])

		Local c:TCategory = New TCategory

		' add values from array to separate fields
		c.Category = ca[0]
		c.Description = ca[1]

		' create a map
		MapInsert(CategoryMap, c.Category, c)

	End Function

	' retrieves a single TCategory Object from a given key
	Function Read:TCategory(Category:String)

		Return TCategory(MapValueForKey(CategoryMap, Category))

	End Function

End Type

Type TStarData

	Global StarDataMap:TMap = CreateMap()

	Global rows:Int = 90
	Global cols:Int = 8

	Field StarType:String
	Field Temperature:String
	Field MagAbs:String
	Field Radius:String
	Field Mass:String
	Field WaveLength:String
	Field Correction1:String
	Field Correction2:String

	' Read StarData Field values
	Function Init()

		' Array to hold Data field values
		Local sd:String[cols]

		' access Data field
		RestoreData StarData

		' cycle all rows
		For Local i:Int = 0 To rows - 1

			' cycle all columns
			For Local j:Int = 0 To cols - 1
		
				' store to array
				ReadData sd[j]

			Next

			' create a new TStarData object
			TStarData.Create(sd)
		
		Next

	End Function

	' create a single TStarData Object
	Function Create(sd:String[])

		Local s:TStarData = New TStarData

		' add values from array to separate fields
		s.StarType = sd[0]
		s.Temperature = sd[1]
		s.MagAbs = sd[2]
		s.Radius = sd[3]
		s.Mass = sd[4]
		s.WaveLength = sd[5]
		s.Correction1 = sd[6]
		s.Correction2 = sd[7]

		' create a map
		MapInsert(StarDataMap, s.StarType, s)

	End Function

	' retrieves a single TStarData Object from a given key
	Function Read:TStarData(StarType:String)

		Return TStarData(MapValueForKey(StarDataMap, StarType))

	End Function

End Type

' single Star System
Type TSystem

	Global SystemSpectralType:String[] = ["O", "B", "A", "F", "G", "K", "M", "W"]
	Global SystemSpectralRoman:String[] = ["I", "II", "III", "IV", "V"]
	
	'Global SystemSpectralDistribution:Float[] = [0.0, 0.02, 0.02, 0.05, 0.16, 0.26, 0.26, 0.22, 0.01, 0.0]   ' realistic
	Global SystemSpectralDistribution:Float[] = [0.0, 0.14, 0.14, 0.14, 0.14, 0.14, 0.14, 0.14, 0.14, 0.0]   ' test
	
	Global PlanetEvaluationDistribution:Int[] = [10, 10, 10, 5, 25, 8, 15, 8, 5, 24, 10]
	Global SpectralDistribution:Float[10]
	Global StarData:String[100, 8]
	Global StarCorrection:String[9, 8]
	Global Masses:Int[10, 3]

	Global syllables:String = "..lexegezacebisousesarmaindirea.eratenberalavetiedorquanteisrion..lexegezacebisousesarmaindirea.eratenberalavetiedorquanteisrion"
	Global prefix:String[] = ["Alpha", "Beta", "Chi", "Delta", "Epsilon", "Eta", "Gamma", "Iota", "Kappa", "Lambda", "My", "Ny", "Omega", "Omicron", "Phi", "Pi", "Psi", "Rho", "Sigma", "Tau", "Theta", "Xi", "Ypsilon", "Zeta"]
	Global nameseed:Int[3]
	
	Const PlanetAlbedoAlpha:Double = 0.33					' Alpha Albedo (Star -> outer Atmosphere -> Space reflection) in %
	Const PlanetAlbedoBeta:Double = 0.04					' Beta Albedo (Planet Surface -> inner Atmosphere reflection) in %
	Const GreenhouseEffect:Double = 24.0					' Greenhouse Effekt in K
	Const MinimumHZTemperature:Double = -50					' Minimum Temperature for habitable planet in °C
	Const OptimumHZTemperature:Double = 0.0					' Optimal Temperature for habitable planet in °C
	Const MaximumHZTemperature:Double = 50					' Maximum Temperature for habitable planet in °C
	Const MinimumGForce:Double = 0.5						' Minimum G-Force for habitable planet in g	
	Const MaximumGForce:Double = 5.0						' Maximum G-Force for habitable planet in g	
	Const MinimumAtmoPressure:Double = 0.125				' Minimum atmospheric Pressure for habitable planet in bar
	Const MaximumAtmoPressure:Double = 2.0					' Maximum atmospheric Pressure for habitable planet in bar
	Const MinimumRotationSpeed:Double = 0.25				' Minimum rotation speed for habitable planet in days
	Const MaximumRotationSpeed:Double = 2.25				' Maximum rotation speed for habitable planet in days
	Const MinimumDensity:Double = 2.5						' Minimum density for habitable planet in g/cm³
	Const MaximumDensity:Double = 7.0						' Maximum density for habitable planet in g/cm³

	Const AstronomicalUnitKM:Double = 149597870.0			' Astronomical Unit in Kilometer
	Const AstronomicalUnitM:Double = 149597870000.0			' Astronomical Unit in Meter
	Const SunRadiusKM:Double = 696342.0						' Sun Radius in Kilometer
	Const SunRadiusM:Double = 696342000.0					' Sun Radius in Meter
	Const SunLuminance:Double = 384699438278147000000.0		' Sun Luminance in Watt/second
	Const SunTemperature:Double = 5796.0					' Sun Temperature in Kelvin
	Const CelsiusKelvin:Double = 273.3						' 0°C in Kelvin
	Const Boltzmann:Double = 0.000000056704					' Boltzmann Constant
	Const Parsec:Double = 3.261564							' Lightyears in Parsec
	Const Earthradius:Double = 6378160000.0					' Earth radius in cm
	Const SunPlane:Double = 8750491646.3441					' Sun Plane in km²
	Const EarthPlane:Double = 511211580303994000000.0		' Earth plane in cm²
	Const EarthGForce:Double = 9.80665						' Earth G-Force in m/sec²
	Const SunMass:Double = 1.989 * (10 ^ 30)				' Sun Mass in KG (x10^30)
	Const SunPower:Double = 30.0							' Sun Mass Power (Mass = Mass x 10^SunPower)
	Const SunDensity:Double = 1.408							' Sun Densitys in g/cm³
	Const LightSpeed:Double = 299792458						' Lightspeed in m/sec
	Const LightYearM:Double = 9460720472580800.0			' Lightyear in Meter
	Const LightYearKM:Double = 9460720472580.800			' Lightyear in Kilometer
	Const LightYearAU:Double = 0.0000158125240496817		' Lightyear in AU
	Const LightYearsAUs:Double = 63241010.534313088816037	' AUs in 1 Light Years
	Const GravitationalConstant:Double = 0.0000000000667384	' Gravitational Constant
	Const ChandrasekharLimit:Double = 1.44					' Supernova Limit in Sun Masses
	Const TolmanOppenHeimerVolkoff:Double = 3.0				' Supernova Limit in Sun Masses
	Const MMWBoltzmann:Double = 1.38 * 10 ^ -23				' MMW Boltzmann Constant in Joule/Kelvin

	Field id:Int = 1										' Unique Star ID
	Field seed:Int = 42										' Unique global star seed
	Field Position:Double[3]								' Star Position XYZ in Lightyears
	Field RGB:Int[] = [255, 255, 255]						' Star Color
	Field Title:String										' Star Name

	' Star Spectrum
	Field SpectralType:String								' Spectral Class of Star
	Field Subclass:String									' Spectral Subclass of Star (0-9)
	Field LuminanceClass:Int								' Spectral Luminance of Star (1-5)
	Field Giant:Int											' Star Giant Flag
	Field SpectralID:Int									' Star Spectral ID
	Field SpectralClass:String								' complete Spectral description
	

	' Star calculations
	Field Distance:Double									' distance from 0,0,0 to star in Lightyears
	Field AbsoluteMagnitude:Double							' Absolute Magintude of Star
	Field BolometricMagnitude:Double						' Bolometric Magintude of Star
	Field TemperatureCalculated:Int							' Calculated Temperature in K
	Field TemperatureTypical:Double							' Typical Temperature in K
	Field TemperatureEstimated:Double						' Estimated Temperature in K
	Field RadiusCalculated:Double							' Calculated Radius compared to Sun
	Field RadiusTypical:Double								' Typical Radius compared to Sun
	Field RadiusEstimatedSun:Double							' Estimated Radius compared to Sun
	Field RadiusEstimatedKM:Double							' Estimated Radius in KM
	Field RadiusEstimatedAU:Double							' Estimated Radius in AU
	Field SunRadii:Double									' Radius factor compared to Sun
	Field MassCalculated:Double								' Calculated Mass compared to Sun
	Field MassTypical:Double								' Typical Mass compared to Sun
	Field MassEstimated:Double								' Estimated Mass compared to Sun
	Field SunMasses:Double									' Mass factor compared to sun
	Field DensitySun:Double									' Density compared to Sun
	Field Density:Double									' Density in g/cm³
	Field Infrared:Double									' Infrared Radiation compared to Sun
	Field Luminance:Double									' Total Luminance compared To Sun
	Field LuminanceWatt:Double								' Total Luminance in Watt
	Field PeakRadiation:Double								' Star's peak radiation in nm
	Field PeakVisible:String								' Star's peak radiation in visible Light
	Field Gravity:Double									' Gravity of Star in m/s
	Field CorrectedHeatOutput:Double						' Corrected Infrared Heat Output of Star
	Field MegaWattPerSquareMeter:Double						' Radiation in Megawatt per m²
	Field Watt:Double										' Watt at Current distance
	Field StarSurface:Double								' Star Surface in m²
	Field SpectralLines:String								' Spectral Lines

	' Star Details
	Field SurfaceTemperature:Double							' Temperature of Star Surface in K
	Field Radius:Double										' Radius of Star in km
	Field Mass:Double										' Estimated Mass in kg
	Field GravityG:Double									' Gravity of Star in g
	Field Metallicity:Double								' Metallicity
	Field Lifetime:Double									' Total lifetime of Star in Years
	Field Age:Double										' Current lifetime of Star in Years
	Field Death:Double										' Death of Star in Years
	Field LifetimePassed:Double								' Star lifetime passed in %
	Field Fate:String										' Star fate after death
	Field StarOldEnough:Int									' Flag If star is old enough For life
	Field SpectrumSuitable:Int								' Flag If star spectrum is suitable For life
	Field MetallicitySuitable:Int							' Flag If metallicity is high enough For life
	Field MinimumPlanetDistance:Double						' Minimum distance of a planet before it gets too hot
	Field EscapeVelocity1AU:Double							' Escape Velocity at 1 AU Distance
	Field PopulationType:String								' Star is Member of Population I/II/III
	
	Field JetRotation:Double[]								' Rotation Angles of Neutron Star
	Field JetSpeed:Double									' Rotation Speed of Neutron Star
	Field Dwarfmode:Int										' White Dwarf class

	Field HZMinAverageTemperature:Double					' Minimum HZ Average Temperature
	Field HZMinSurfaceEnergy:Double							' Minimum HZ Surface Energy
	Field HZMinSurfaceEnergyCorrected:Double				' Corrected Minimum HZ Surface Energy
	Field HZMinSolarConstant:Double							' Minimum HZ Solar Constant
	Field HZMinMinimumPlanetDistanceAU:Double				' Minimum HZ Planet Distance in AU
	Field HZMinMinimumPlanetDistanceKM:Double				' Minimum HZ Planet Distance in KM


	Field HZOptAverageTemperature:Double					' Optimum HZ Average Temperature
	Field HZOptSurfaceEnergy:Double							' Optimum HZ Surface Energy
	Field HZOptSurfaceEnergyCorrected:Double				' Corrected Optimum HZ Surface Energy
	Field HZOptSolarConstant:Double							' Optimum HZ Solar Constant
	Field HZOptMinimumPlanetDistanceAU:Double				' Optimum HZ Planet Distance in AU
	Field HZOptMinimumPlanetDistanceKM:Double				' Optimum HZ Planet Distance in KM

	Field HZMaxAverageTemperature:Double					' Maximum HZ Average Temperature
	Field HZMaxSurfaceEnergy:Double							' Maximum HZ Surface Energy
	Field HZMaxSurfaceEnergyCorrected:Double				' Corrected Maximum HZ Surface Energy
	Field HZMaxSolarConstant:Double							' Maximum HZ Solar Constant
	Field HZMaxMinimumPlanetDistanceAU:Double				' Maximum HZ Planet Distance in AU
	Field HZMaxMinimumPlanetDistanceKM:Double				' Maximum HZ Planet Distance in KM

	Field HabitableZoneMinimum:Double						' Minimum distance For habitable zone
	Field HabitableZoneOptimum:Double						' Optimum distance For habitable zone
	Field HabitableZoneMaximum:Double						' Maximum distance For habitable zone

	Field Planet:TPlanet[21]								' Star System Planet Array

	Method Init()

		Local i:Int

		For i = 1 To 9

			SpectralDistribution[i] = SpectralDistribution[i - 1] + SystemSpectralDistribution[i]

		Next

		TStarData.Init()
		TCategory.Init()

		RestoreData Correction
		For i = 0 To 7

			ReadData StarCorrection[i, 0], StarCorrection[i, 1], StarCorrection[i, 2], StarCorrection[i, 3], StarCorrection[i, 4], StarCorrection[i, 5], StarCorrection[i, 6], StarCorrection[i, 7]

		Next

		RestoreData Masses
		For i = 0 To 9

			ReadData Masses[i, 0], Masses[i, 1], Masses[i, 2]

		Next

	End Method
	
	Method SysRnd:Double(mn:Double = 0.0, mx:Double = 1.0, secondseed:Int = 13)
	
		Local r:Double
		Local n:Int = id + secondseed * 57 + seed
	
		n = n Shl Int(Floor((13 ^ n)))
		
		r = (1.0 - ((n * (n * n * 15731 + 789221) + 1376312589) & $7fffffff) / 1073741824.0)
		
		Return Double(Normalize(r, -1.0, 1.0, mn, mx))
		
	End Method

	Method SysRand:Int(mn:Int = 0, mx:Int = 1, secondseed:Int = 13)
	
		Local r:Double
		Local n:Int = id + secondseed * 57 + seed
	
		n = n Shl Int(Floor((13 ^ n)))
		
		r = (1.0 - ((n * (n * n * 15731 + 789221) + 1376312589) & $7fffffff) / 1073741824.0)
		
		r = (1.0 + r) / 2.0
		
		Local rg:Int = mx - mn
		If rg > 0 Then Return Int(r * (1 + rg)) + mn
	
		Return Int(r * (1 - rg)) + mx
		
		
		'Return RoundInt(Normalize(((1.0 + r) / 2.0), 0, 1, mn, mx))
		
		
		'Return Normalize(RoundInt(r), -1.0, 1.0, mn, mx)
		
	End Method

	Method NewRandomSystem()

		' determine Spectral Type first
		SpectralType=CalcSpectralType()
		SubClass = SysRand(0, 9, sysid + 1)
		LuminanceClass = SysRand(1, 5, sysid + 2)
		Giant = CalcGiant()
		
		JetRotation = [SysRnd(-90, 90, sysid + 3), SysRnd(-90, 90, sysid + 4)]
		JETSPEED = SysRnd(100, 250, sysid + 5)
		Dwarfmode = SysRand(0, 3, sysid + 6)
				
		'System.SpectralType = "G"
		'System.SubClass = 2
		'System.LuminanceClass = 5
		'System.Giant = 0
				
		SpectralClass = CalcSpectralClass()
				
		'Print SpectralClass
		
		Title = CreateName()
		
	End Method

	Method CalcSystem()
	
		'SpectralClass = CalcSpectralClass()
		
		Local temp:TStarData = TStarData.Read(SpectralType + SubClass)
		
		For Local i:Int = 0 To 7
				
			If SystemSpectralType[i] = SpectralType Then SpectralID = i
		
		Next
						
		' Read stored data from "Correction" Table To a more readable format
		Local CR_Correction1:Double = GetStarCorrection(String(SpectralType))[0]
		Local CR_Correction2:Double = GetStarCorrection(String(SpectralType))[1]
		Local CR_Wavelength:Double = GetStarCorrection(String(SpectralType))[2]
		'Local CR_Spectrum:Double=GetStarCorrection(String(SpectralType))[3]
		'Local CR_SpectralLines:Double=GetStarCorrection(String(SpectralType))[4]
		
		' Calculate Star Variables
		Position = [SysRnd(-5000, 5000), SysRnd(-5000, 5000), SysRnd(-5000, 5000)]
		Distance = CalcDistance()
		AbsoluteMagnitude = Double(temp.MagAbs) - (Giant * LuminanceClass ^ CR_Correction2)
		BolometricMagnitude = AbsoluteMagnitude + CR_Correction1
		TemperatureCalculated = (2.898 * (10 ^ 6)) / CR_Wavelength
		TemperatureTypical = Double(temp.Temperature)

		TemperatureEstimated = ((TemperatureCalculated + TemperatureTypical) / 2.0) * SysRnd(0.7, 1.3)
		Luminance = 10 ^ ((4.83 - AbsoluteMagnitude) / 2.5) * SysRnd(0.9, 1.1)
		RadiusCalculated = Sqr(Luminance) / ((TemperatureEstimated * 1.0 / SunTemperature) ^ 2) * SysRnd(0.9, 1.1)
		RadiusTypical = Double(temp.Radius) * SysRnd(0.9, 1.1)
		
		RadiusEstimatedSun = CalcRadius(RadiusCalculated, RadiusTypical)
		RadiusEstimatedKM = RadiusEstimatedSun * SunRadiusKM
		RadiusEstimatedAU = RadiusEstimatedKM * 1.0 / AstronomicalUnitKM
		MassCalculated = Luminance ^ (1.0 / 3.5)
		MassTypical = Double(temp.Mass)
		MassEstimated = (MassCalculated + MassTypical) / 2.0
		Mass = MassEstimated * SunMass
		DensitySun = MassEstimated * 1.0 / RadiusEstimatedSun
		'DensitySun=Density*1.0/SunDensity
		Density = DensitySun * SunDensity
		
		Infrared = Double(temp.Correction2)
		LuminanceWatt = 4 * Pi * Boltzmann * ((RadiusEstimatedKM * 1000) ^ 2) * (TemperatureEstimated ^ 4) * Infrared
		PeakRadiation = Double(temp.Wavelength)
		PeakVisible = GetPeakVisible(PeakRadiation)
		Gravity = Sqr((2 * GravitationalConstant * Mass) / (RadiusEstimatedKM * (10 ^ 9)))
		GravityG = Gravity * 1.0 / EarthGForce
		CorrectedHeatOutput = LuminanceWatt * Infrared
		SpectralLines = GetSpectralLines(SpectralType)
		
		SurfaceTemperature = ((TemperatureCalculated + TemperatureTypical) / 2.0) * SysRnd(0.95, 1.05)
		MegaWattPerSquareMeter = 5.67 * 10 ^ -7 * SurfaceTemperature ^ 4 / 10000000.0
		Radius = RadiusEstimatedKM
		SunRadii = Radius * 1.0 / SunRadiusKM
		
		StarSurface = 4 * Pi * (Radius * 1000) ^ 2
		Lifetime = (10 ^ 10) * (MassEstimated ^ -2.5)
		
		Age = CalcAge()
		Death = Lifetime - Age
		LifetimePassed = Age * 1.0 / Lifetime * 100
		Metallicity = Log10(Age * 1.0 / (Lifetime / 2.0111974))
		MetallicitySuitable = CalcMetallicityLife()
		Fate = CalcFate()
		StarOldEnough = CalcOldEnough()
		EscapeVelocity1AU = CalcEscapeVelocity()
		SpectrumSuitable = CalcSpectrumSuitable()
		
		SunMasses = Mass * 1.0 / SunMass
		
		' special Case White Dwarf
		Local wf:Float = 1.0
		If SpectralType = "W" Then wf = 0.1
		
		HZMinAverageTemperature = CelsiusKelvin - GreenhouseEffect - MinimumHZTemperature
		HZMinSurfaceEnergy = (HZMinAverageTemperature ^ 4) * Boltzmann
		HZMinSurfaceEnergyCorrected = (HZMinSurfaceEnergy * 1.0 / (1 - PlanetAlbedoAlpha - PlanetAlbedoBeta))
		HZMinSolarConstant = HZMinSurfaceEnergyCorrected * 4
		HZMinMinimumPlanetDistanceAU = Sqr(0.25 * ((LuminanceWatt * 1.0 / Pi) / HZMinSolarConstant)) / AstronomicalUnitM * wf ' 0.25
		HZMinMinimumPlanetDistanceKM = HZMinMinimumPlanetDistanceAU * AstronomicalUnitKM * 1.0 / 10000000 * wf
		
		HZOptAverageTemperature = OptimumHZTemperature + CelsiusKelvin - GreenhouseEffect
		HZOptSurfaceEnergy = (HZOptAverageTemperature ^ 4) * Boltzmann
		HZOptSurfaceEnergyCorrected = (HZOptSurfaceEnergy * 1.0 / (1 - PlanetAlbedoAlpha - PlanetAlbedoBeta))
		HZOptSolarConstant = HZOptSurfaceEnergyCorrected * 4
		HZOptMinimumPlanetDistanceAU = Sqr(0.5 * ((LuminanceWatt * 1.0 / Pi) / HZOptSolarConstant)) / AstronomicalUnitM * wf ' 0.5
		HZOptMinimumPlanetDistanceKM = HZOptMinimumPlanetDistanceAU * AstronomicalUnitKM * 1.0 / 10000000 * wf
						
		HZMaxAverageTemperature = CelsiusKelvin - GreenhouseEffect - MaximumHZTemperature
		HZMaxSurfaceEnergy = (HZMaxAverageTemperature ^ 4) * Boltzmann
		HZMaxSurfaceEnergyCorrected = (HZMaxSurfaceEnergy * 1.0 / (1 - PlanetAlbedoAlpha - PlanetAlbedoBeta))
		HZMaxSolarConstant = HZMaxSurfaceEnergyCorrected * 4
		HZMaxMinimumPlanetDistanceAU = Sqr(0.75 * ((LuminanceWatt * 1.0 / Pi) / HZMaxSolarConstant)) / AstronomicalUnitM * wf ' 0.75
		HZMaxMinimumPlanetDistanceKM = HZMaxMinimumPlanetDistanceAU * AstronomicalUnitKM * 1.0 / 10000000 * wf

		MinimumPlanetDistance = CalcMinimumDistance() * wf

	End Method

	Method CalcSpectralType:String()

		Local s:String
		Local star:Float = SysRnd()
		
		'Print id + ": " + star

		For Local i:Int = 0 To 7
		
			If star >= SpectralDistribution[i] And star <= SpectralDistribution[i + 1] Then s = SystemSpectralType[i]

		Next
		
		Return s

	End Method

	Method CalcSpectralClass:String()

		Local addendum:String = LOCALE.Get("SOLAR_MAINSEQUENCE")'"Main Sequence"

		If Giant = 1 Then addendum = LOCALE.Get("SOLAR_GIANT")'"Giant"
		If SpectralType = "W" Then
		
			addendum = LOCALE.Get("SOLAR_WHITEDWARF")'"White Dwarf"
			
			addendum = WHITEDWARF_MODES[Dwarfmode]
			
		EndIf
		
		'Print LuminanceClass
		
		Return SpectralType + Subclass + "-" + SystemSpectralRoman[LuminanceClass - 1] + " [" + addendum + "]"

	End Method

	Method CalcGiant:Int()
		
		If SpectralType <> "G" And SpectralType <> "W" Then Return SysRand(0, 1)

	End Method

	Method CalcDistance:Float()

		Local x:Float = Position[0]
		Local y:Float = Position[1]
		Local z:Float = Position[2]

		Return Sqr(x * x + y * y + z * z)

	End Method

	Method CalcMinimumDistance:Double()

		'If SpectralType="W" Then Return ((5*RadiusEstimatedSun)^2)/2.0 Else Return Sqr(HZMinMinimumPlanetDistanceAU)/2.0
		If SpectralType = "W" Then Return ((5 * RadiusEstimatedSun) ^ 2) / 2.0 Else Return Sqr(RadiusEstimatedSun / 2.0)

	End Method

	Method CalcRadius:Float(value1:Float, value2:Float)

		If value1 > value2 Then Return value1 Else Return value2

	End Method

	Method CalcAge:Double()

		Local age1:Double
		Local age2:Double

		If RadiusCalculated > RadiusTypical Then

			age1 = 1 - (RadiusTypical / 2.0 / RadiusCalculated)

		Else

			age1 = RadiusCalculated / 2.0 / RadiusTypical

		EndIf

		If age1 * Lifetime > (10 ^ 10) Then

			age2 = (10 ^ 10) * SysRnd(0.5, 1.1)

		Else

			age2 = (age1 * Lifetime) * (18.36 * LuminanceClass / 100.0)

		EndIf

		Return age2

	End Method

	Method CalcFate:String()

		If MassEstimated >= 0 And MassEstimated < 8 Then Return "White Dwarf"
		If MassEstimated >= 8 And MassEstimated < 15 Then Return "Pulsar"
		If MassEstimated >= 15 And MassEstimated < 25 Then Return "Neutron Star"
		If MassEstimated >= 25 Then Return "BLACK Hole"

	End Method

	Method CalcOldEnough:Int()

		If Lifetime > 10 ^ 9 And Age > 10 ^ 9 Then Return True Else Return False

	End Method

	Method CalcSpectrumSuitable:Int()

		If PeakVisible <> "Ultraviolet" Then Return True Else Return False

	End Method

	' Calculate Escape velocity in Meter/second at a given AU distance
	Method CalcEscapeVelocity:Double(au:Double = 1.0)
	
		If SpectralType = "W" Then Return Sqr((2 * GravitationalConstant * Mass) / (au * AstronomicalUnitM * 1000)) * 10
		
		Return Sqr((2 * GravitationalConstant * Mass) / (au * AstronomicalUnitM * 1000))

	End Method

	Method CalcMetallicityLife:Int()

		If Metallicity > - 0.5 Then Return True Else Return False

	End Method

	' pull information about typical Star properties from data Set
	Method GetStarCorrection:Float[] (startype:String)

		Local r:Float[] = Null

		For Local i:Int = 0 To 8

			If startype = StarCorrection[i, 0] Then

				r = [Float(StarCorrection[i, 1]), Float(StarCorrection[i, 2]), Float(StarCorrection[i, 3]), Float(StarCorrection[i, 4]), Float(StarCorrection[i, 5]), Float(StarCorrection[i, 6])]

			EndIf

		Next

		Return r

	End Method

	Method GetPeakVisible:String(nm:Float)

		Local txt:String = "-"

		For Local i:Int = 0 To 7

			If nm >= Float(StarCorrection[i, 4]) And nm < Float(StarCorrection[i, 5]) Then txt = StarCorrection[i, 6]

		Next

		Return txt

	End Method

	Method GetSpectralLines:String(startype:String)

		Local txt:String = "-"

		For Local i:Int = 0 To 8

			If startype = StarCorrection[i, 0] Then txt = StarCorrection[i, 7]

		Next

		Return txt

	End Method

	Method CalcTemperatureAt:Double(dist:Float)
			
		Return ((CalcSolarConstantAt(dist) / (4 * Boltzmann)) ^ 0.25) - CelsiusKelvin
		
	End Method
	
	Method CalcSolarConstantAt:Double(dist:Float)
	
		Local t1:Double = SurfaceTemperature ^ 4 * Boltzmann * (Radius * 1000.0) ^ 2
		Local t2:Double = (dist * 1000000000) ^ 2
		
		Return t1 / t2
				
	End Method
	
	Method CalcZoneDistance:Double(solarconstant:Float = 1367.0)
	
		Local watt:Double = 4 * Pi * Boltzmann * (Radius * 1000) ^ 2 * (SurfaceTemperature ^ 4)
		Local dist:Double = Sqr(0.25 * ((watt / Pi) / solarconstant)) / AstronomicalUnitM
		
		Return dist
		
	End Method

	Method CalcMinimumHabitableZoneDistance:Double()
	
		Return CalcZoneDistance(1800)
	
	End Method

	Method CalcOptimumHabitableZoneDistance:Double()
	
		Return CalcZoneDistance(1367)
	
	End Method

	Method CalcMaximumHabitableZoneDistance:Double()
	
		Return CalcZoneDistance(900)
	
	End Method

	Method CalcBlackHoleRadius:Double()
	
		Return 2.0 * GravitationalConstant * Mass / (LightSpeed ^ 2)
	
	End Method
	
' ----------------------------------------------------------------------------
' shuffle planet name seed
' ----------------------------------------------------------------------------
Method Tweakseed()
    
	Local temp:Int = ((nameseed[0] + nameseed[1] + nameseed[2])) Mod $10000
	nameseed[0] = nameseed[1]
	nameseed[1] = nameseed[2]
	nameseed[2] = temp
    
End Method



' ----------------------------------------------------------------------------
' Create Random Planet name (elite style)
' ----------------------------------------------------------------------------
Method CreateName:String()
	
	Local longnameflag:Int = nameseed[0] & $80
	Local planetname:String = ""
	Local n:Int, d:Int
	
	Local pf:Int = nameseed[0] Mod prefix.length - 1
    
	For n = 0 To 3
        
		d = ((nameseed[2] Shr 8) & $1f) Shl 1
		Tweakseed()
		        
		If n < 3 Or longnameflag Then
			
			planetname = planetname + Mid(syllables, 1 + d, 2)
			planetname = Replace(planetname, ".", "")
            
		EndIf
        
	Next
    
	planetname = Upper(Mid(planetname, 1, 1)) + Mid(planetname, 2, Len(planetname) - 1)
	
	Title = prefix[SysRand(0, prefix.Length - 1)] + " " + planetname
    
	Return planetname
	
	'Return planetname
    
End Method



' ----------------------------------------------------------------------------
' Set a planet
' ----------------------------------------------------------------------------
Method SetStar(num:Int = 0)
    
	Local i:Int
    
	'nameseed[0] = $5A4A | seed
	'nameseed[1] = $0248 | seed
	'nameseed[2] = $B753 | seed

	nameseed[0] = $5A4A
	nameseed[1] = $0248
	nameseed[2] = $B753
	
	'nameseed[0] = SysRand($7FFF, $FFFF)
	'nameseed[1] = SysRand($7FFF, $FFFF)
	'nameseed[2] = SysRand($7FFF, $FFFF)
	
	For i = 0 To num - 1
        
		Tweakseed()
		Tweakseed()
		Tweakseed()
		Tweakseed()
        
	Next
    
End Method

End Type

' Spectral Class / Correction1 / Correction2 / Wavelength
#Correction
DefData "W", -4.00, 1.000, 10, 780, 999, "Infrared", "Helium"
DefData "O", -4.00, 0.733, 100, 0, 380, "Ultraviolet", "Ionized Helium"
DefData "B", -2.00, 1.129, 200, 380, 430, "Violet", "Neutral Helium"
DefData "A", -0.30, 1.479, 300, 430, 490, "Blue", "Hydrogen"
DefData "F", -0.15, 1.585, 400, 490, 570, "Green", "Ionized Metals"
DefData "G", -0.40, 1.665, 500, 570, 600, "Yellow", "Ionized Metals"
DefData "K", -0.80, 1.725, 600, 600, 640, "Orange", "Neutral Metals"
DefData "M", -2.00, 1.829, 800, 640, 780, "RED", "Molecules"
DefData "R", -2.00, 2.000, 900, 999, 999, "", ""

' Orbit / MinMass / MaxMass
#OrbitMass
DefData 0, 21, 29
DefData 1, 21, 28
DefData 2, 21, 26
DefData 3, 21, 24
DefData 4, 21, 28
DefData 5, 21, 29
DefData 6, 21, 26
DefData 7, 21, 24
DefData 8, 21, 23
DefData 9, 21, 22

' Age / Metallicity
#Metallicity
DefData 11750000000, 0.0004
DefData 2400000000, 0.004
DefData 1450000000, 0.008
DefData 900000000, 0.02
DefData 550000000, 0.05

' from Sunmasses / To Sunmasses / Fate
#Fate
DefData 0, 8, "White Dwarf"
DefData 8, 15, "Pulsar"
DefData 15, 25, "Neutron Star"
DefData 25, 999, "BLACK Hole"

' Planet Orbit / MinMass / MaxMass (1.0x10^x)
#Masses
DefData 0, 21, 29
DefData 1, 21, 28
DefData 2, 21, 26
DefData 3, 21, 24
DefData 4, 21, 28
DefData 5, 21, 29
DefData 6, 21, 26
DefData 7, 21, 24
DefData 8, 21, 23
DefData 9, 21, 22


' Category / Common Description
#Category
DefData "Very Hot Dwarf", "Molten Dwarf"
DefData "Very Hot Rock", "Molten"
DefData "Very Hot Super Earth", "Molten"
DefData "Very Hot Gas", "Hot Neptune"
DefData "Very Hot Gas Giant", "Hot Jupiter"
DefData "Very Hot Brown Dwarf", "Brown Dwarf"

DefData "Hot Dwarf", "Mercurian Dwarf"
DefData "Hot Rock", "Mercurian"
DefData "Hot Super Earth", "Super Mercury"
DefData "Hot Gas", "Hot Neptune"
DefData "Hot Gas Giant", "Hot Jupiter"
DefData "Hot Brown Dwarf", "Brown Dwarf"

DefData "Arid Dwarf", "Martian"
DefData "Arid Rock", "Tattoineian"
DefData "Arid Super Earth", "Duneian"
DefData "Arid Gas", "None"
DefData "Arid Gas Giant", "None"
DefData "Arid Brown Dwarf", "Brown Dwarf"

DefData "Temperate Dwarf", "None"
DefData "Temperate Rock", "Earthlike"
DefData "Temperate Super Earth", "Earthlike"
DefData "Temperate Gas", "None"
DefData "Temperate Gas Giant", "None"
DefData "Temperate Brown Dwarf", "Brown Dwarf"

DefData "Cold Dwarf", "None"
DefData "Cold Rock", "None"
DefData "Cold Super Earth", "None"
DefData "Cold Gas", "None"
DefData "Cold Gas Giant", "None"
DefData "Cold Brown Dwarf", "Brown Dwarf"

DefData "Frozen Dwarf", "Ice Dwarf"
DefData "Frozen Rock", "Hothian"
DefData "Frozen Super Earth", "Ice Planet"
DefData "Frozen Gas", "None"
DefData "Frozen Gas Giant", "None"
DefData "Frozen Brown Dwarf", "Brown Dwarf"

DefData "Very Cold Dwarf", "Ice Dwarf"
DefData "Very Cold Rock", "None"
DefData "Very Cold Super Earth", "None"
DefData "Very Cold Gas", "None"
DefData "Very Cold Gas Giant", "None"
DefData "Very Cold Brown Dwarf", "Brown Dwarf"


' Radius from / Radius To / Category
#Size
DefData 0, 5000, "Dwarf"
DefData 5000, 8000, "Rock"
DefData 8000, 12000, "Super Earth"
DefData 12000, 20000, "Gas"
DefData 20000, 80000, "Gas Giant"
DefData 80000, 1000000, "Brown Dwarf"

' Temp from / Temp To / Category
#Temperature
DefData - 274, -125, "Very Cold"
DefData - 125, -75, "Frozen"
DefData - 75, -25, "Cold"
DefData - 25, 25, "Temperate"
DefData 25, 75, "Arid"
DefData 75, 500, "Hot"
DefData 500, 9999, "Very Hot"

' from / To / Weather
#Weather
DefData 0, 10, "None"
DefData 10, 40, "Calm"
DefData 40, 80, "Moderate"
DefData 80, 120, "Dynamic"
DefData 120, 250, "Violent"
DefData 250, 10000, "Inferno"

' Mass1 / Mass2 / Element / Formula
#Atmosphere
DefData 0.00, 1.00, "Hydrogen", "H2"
DefData 1.00, 1.40, "Helium", "He2"
DefData 1.40, 1.80, "Methane", "CH4"
DefData 1.80, 2.50, "Ammonia", "NH3"
DefData 2.50, 3.50, "Carbon Monoxide", "CO"
DefData 3.50, 4.50, "Nitrogen", "N2"
DefData 4.50, 5.50, "Oxygen", "O2"
DefData 5.00, 5.50, "Carbon Dioxide", "CO2"
DefData 5.50, 6.00, "Nitrogen Dioxide", "NO2"
DefData 6.00, 9.99, "Sulphur Dioxide", "SO2"

' Temp1 / Temp2 / Element / Formula
#Hydrosphere
DefData - 300, -260.00, "Hydrogen", "H2"
DefData - 260, -220.00, "Helium", "He2"
DefData - 220, -180.00, "Nitrogen", "N2"
DefData - 180, -150.00, "Methanol", "CH3OH"
DefData - 150, -100.00, "Ethanol", "C2H6O"
DefData - 100, -60.00, "Ammonia", "NH3"
DefData - 60, 60.00, "Water", "H2O"
DefData 60, 150.00, "Hydrogen Chloride", "HCl"
DefData 150, 250.00, "Nitrogen Dioxide", "NO2"
DefData 250, 100000.00, "None", "None"

' Mass1 / Mass2 / Element / Formula
#Lithosphere
DefData 0, 10, "Lithium", "Li"
DefData 10, 12, "Magnesium", "Mg"
DefData 12, 13, "Aluminium", "Al"
DefData 13, 22, "Titanium", "Ti"
DefData 22, 24, "Chromium", "Cr"
DefData 24, 26, "Iron", "Fe"
DefData 26, 27, "Cobalt", "Co"
DefData 27, 28, "Nickel", "Ni"
DefData 28, 29, "Copper", "Cu"
DefData 29, 30, "Zinc", "Zn"
DefData 30, 42, "Molybdenum", "Mo"
DefData 42, 47, "Silver", "Ag"
DefData 47, 50, "Tin", "Sn"
DefData 50, 51, "Antimony", "Sb"
DefData 51, 61, "Promethium", "Pm"
DefData 61, 74, "Tungsten", "W"
DefData 74, 78, "Platinum", "Pt"
DefData 78, 79, "Gold", "Au"
DefData 79, 80, "Mercury", "Hg"
DefData 80, 82, "Lead", "Pb"
DefData 82, 90, "Uranium", "U"
DefData 90, 93, "Plutonium", "Pu"
DefData 93, 95, "Dopatium", "Dp"
DefData 95, 100, "Danikum", "Dk"
DefData 100, 1000, "None", "None"

' StarType / Temp / AbsMag / Radius / Mass / Wavelength / Correction1 / Correction2
#StarData
DefData "W0", 32500, 14.5, 0.02, 1.3, 10, 0, 0.01
DefData "W1", 30000, 14, 0.017, 1.17, 20, 0, 0.01
DefData "W2", 27500, 13.5, 0.015, 1.045, 30, 0, 0.01
DefData "W3", 25000, 13, 0.013, 0.92, 40, 0, 0.01
DefData "W4", 22500, 12.5, 0.011, 0.795, 50, 0, 0.01
DefData "W5", 20000, 12, 0.009, 0.67, 60, 0, 0.01
DefData "W6", 17500, 11.5, 0.007, 0.545, 70, 0, 0.01
DefData "W7", 15000, 11, 0.005, 0.42, 80, 0, 0.01
DefData "W8", 12500, 10.5, 0.003, 0.295, 90, 0, 0.01
DefData "W9", 10000, 10, 0.001, 0.17, 100, 0, 0.01
DefData "O0", 62000, -6.811602, 19.22, 146.4, 110, -4, 15.91
DefData "O1", 58500, -6.563001, 18.18, 130, 120, -4, 16.94
DefData "O2", 55000, -6.3144, 17.14, 113.6, 130, -4, 17.67
DefData "O3", 51500, -6.065799, 16.1, 97.2, 140, -4, 18.09
DefData "O4", 48000, -5.7776, 14.4, 90, 150, -4, 16.61
DefData "O5", 44500, -5.58088, 15, 60, 160, -4, 20.51
DefData "O6", 41000, -5.36795, 12.9, 37, 170, -4, 17.12
DefData "O7", 38000, -5.09703, 11.8, 30, 180, -4, 16.06
DefData "O8", 35800, -4.77652, 10.8, 23, 190, -4, 14.99
DefData "O9", 33000, -4.37762, 9.56, 23.3, 200, -4, 13.01
DefData "B0", 30000, -3.87002, 8.47, 17.5, 210, -3, 11.26
DefData "B1", 25400, -3.05072, 6.56, 14.2, 220, -2.8, 7.41
DefData "B2", 22000, -2.27964, 5.22, 10.9, 230, -2.6, 5.13
DefData "B3", 18700, -1.4955, 4.17, 7.6, 240, -2.4, 3.57
DefData "B4", 17050, -1.287265, 4.115, 6.75, 250, -2.2, 3.77
DefData "B5", 15400, -1.07903, 4.06, 5.9, 260, -2, 3.97
DefData "B6", 14000, -0.77759, 3.81, 5.2, 270, -1.83, 3.77
DefData "B7", 13000, -0.47962, 3.54, 4.5, 280, -1.66, 3.5
DefData "B8", 11900, -0.07828, 3.17, 3.8, 290, -1.49, 3.01
DefData "B9", 10500, 0.32649, 2.96, 3.35, 300, -1.32, 2.81
DefData "A0", 9520, 0.728796, 2.71, 2.9, 310, -1.15, 2.51
DefData "A1", 9230, 1.12998, 2.32, 2.72, 320, -0.98, 1.96
DefData "A2", 8970, 1.42097, 2.12, 2.54, 330, -0.81, 1.74
DefData "A3", 8720, 1.62175, 2.01, 2.36, 340, -0.64, 1.66
DefData "A4", 8460, 1.833445, 1.935, 2.18, 350, -0.47, 1.63
DefData "A5", 8200, 2.04514, 1.86, 2, 360, -0.3, 1.6
DefData "A6", 8025, 2.18757, 1.81, 1.92, 370, -0.29, 1.6
DefData "A7", 7850, 2.33, 1.76, 1.84, 380, -0.27, 1.59
DefData "A8", 7580, 2.52319, 1.71, 1.76, 390, -0.26, 1.58
DefData "A9", 7390, 2.67057, 1.675, 1.68, 400, -0.24, 1.6
DefData "F0", 7200, 2.81795, 1.64, 1.6, 410, -0.23, 1.61
DefData "F1", 7045, 3.052725, 1.55, 1.56, 420, -0.21, 1.51
DefData "F2", 6890, 3.2875, 1.46, 1.52, 430, -0.2, 1.4
DefData "F3", 6665, 3.46235, 1.45, 1.46, 440, -0.18, 1.45
DefData "F4", 6552, 3.549775, 1.445, 1.43, 450, -0.17, 1.51
DefData "F5", 6440, 3.6372, 1.44, 1.4, 460, -0.15, 1.56
DefData "F6", 6320, 3.876655, 1.35, 1.295, 470, -0.63, 1.43
DefData "F7", 6260, 3.9963825, 1.305, 1.2425, 480, -0.6, 1.4
DefData "F8", 6200, 4.11611, 1.26, 1.19, 490, -0.58, 1.36
DefData "F9", 6115, 4.30613, 1.195, 1.12, 500, -0.55, 1.27
DefData "G0", 6030, 4.49615, 1.13, 1.05, 510, -0.53, 1.18
DefData "G1", 5945, 4.67961, 1.075, 1.024, 520, -0.5, 1.11
DefData "G2", 5796, 4.86307, 1, 1, 530, -0.48, 1
DefData "G3", 5815, 5.047975, 0.9565, 0.959, 540, -0.45, 0.95
DefData "G4", 5792, 5.1404275, 0.92475, 0.9395, 550, -0.43, 0.92
DefData "G5", 5770, 5.23288, 0.893, 0.92, 560, -0.4, 0.89
DefData "G6", 5670, 5.346205, 0.884, 0.881, 570, -0.44, 0.9
DefData "G7", 5620, 5.4028675, 0.8795, 0.8615, 580, -0.48, 0.93
DefData "G8", 5570, 5.45953, 0.875, 0.842, 590, -0.52, 0.95
DefData "G9", 5410, 5.730415, 0.8305, 0.816, 600, -0.56, 0.88
DefData "K0", 5250, 6.0013, 0.786, 0.79, 610, -0.6, 0.82
DefData "K1", 5080, 6.2121, 0.788, 0.766, 620, -0.64, 0.85
DefData "K2", 4900, 6.52445, 0.75, 0.742, 630, -0.68, 0.79
DefData "K3", 4730, 6.69182, 0.762, 0.718, 640, -0.72, 0.85
DefData "K4", 4590, 7.13205, 0.692, 0.694, 650, -0.76, 0.72
DefData "K5", 4350, 7.54018, 0.684, 0.67, 660, -0.8, 0.73
DefData "K6", 4205, 7.904735, 0.6625, 0.638, 670, -0.92, 0.7
DefData "K7", 4060, 8.26929, 0.641, 0.606, 680, -1.04, 0.68
DefData "K8", 3955, 8.597485, 0.6335, 0.558, 690, -1.16, 0.68
DefData "K9", 3902, 8.7615825, 0.62975, 0.534, 700, -1.28, 0.69
DefData "M0", 3850, 8.92568, 0.626, 0.51, 710, -1.4, 0.7
DefData "M1", 3720, 9.41912, 0.597, 0.445, 720, -1.52, 0.66
DefData "M2", 3580, 10.0167, 0.553, 0.4, 730, -1.64, 0.58
DefData "M3", 3470, 10.5193, 0.527, 0.35, 740, -1.76, 0.54
DefData "M4", 3370, 11.4447, 0.406, 0.3, 750, -1.88, 0.33
DefData "M5", 3240, 12.3857, 0.334, 0.25, 760, -2, 0.23
DefData "M6", 3050, 13.674, 0.262, 0.207, 770, -2.2, 0.14
DefData "M7", 2940, 14.3898, 0.226, 0.163, 780, -2.4, 0.11
DefData "M8", 2640, 16.1628, 0.166, 0.12, 790, -2.6, 0.06
DefData "M9", 2510, 19.6689, 0.092, 0.1, 800, -2.8, 0.02
DefData "R0", 2500, 10, 0.075, 0.1, 810, -3, 0.01
DefData "R1", 2420, 11, 0.03, 0.09, 820, -3.2, 0.01
DefData "R2", 2280, 12, 0.03, 0.08, 830, -3.4, 0.01
DefData "R3", 2140, 13, 0.03, 0.07, 840, -3.6, 0.01
DefData "R4", 2000, 14, 0.03, 0.06, 850, -3.8, 0.01
DefData "R5", 1860, 15, 0.03, 0.05, 860, -4, 0.01
DefData "R6", 1720, 16, 0.03, 0.04, 870, -4.2, 0.01
DefData "R7", 1580, 17, 0.03, 0.03, 880, -4.4, 0.01
DefData "R8", 1440, 18, 0.03, 0.02, 890, -4.6, 0.01
DefData "R9", 1300, 19, 0.03, 0.01, 900, -4.8, 0.01